## Calculated Members

Calculated measures allows for defining new measures.




### Description

A powerful feature of MDX is defining new measures, known as calculated measures (or calculated members).

They are defined before the SELECT statement :

<pre>
WITH
  MEMBER [Measures].[Diff with United States] AS [Measures].[Amount] - ([Geography].[United States],[Measures].[Amount])
  MEMBER [Geography].[Geo].[Europe I] AS [Geography].[Geo].[Europe] - [Geography].[Geo].[Switzerland]
SELECT
  ...
</pre>

In this example, the new measure allows for defining a value based on a formula. This formula may override any member from any hierarchy as defined by the evaluation context of the formula (e.g., axes, slicer, etc...).

Similarly, calculated members can be declared (and removed) at schema level using the `CREATE / DROP` statements.

<pre>
DROP MEMBER [Measures].[Benefit]
CREATE CALCULATED MEMBER [Measures].[Benefit] AS [Measures].[Sales] - [Measures].[Cost]
</pre>

[Cell Properties](Cell%20Properties.md) allows for defining meta-information attached to calculated members.

Please, check working examples for further explanation.



             
### Examples

           

<div class="ic3-mdx-example-header ic3-mdx-example-header-collapsed" onclick="ic3_mdx_example_click(this)">

#### Simple example

</div>

<div class="ic3-mdx-example-body" style="display:none">

A simple example defining a new measure (i.e., [New Value]) as a constant equal to 2.

**Query**

```
WITH
    MEMBER [Measures].[New Value] AS 2
SELECT
    { [Geography].[Geo].[United States] } ON 0,
    { [Measures].[New Value] } ON 1
FROM
    [Sales]
```

 
**Result**

<table  class="ic3-mdx-result">
<tr>
  <td class="ic3-mdx-result-cell-empty"></td>
  <td class="ic3-mdx-result-cell-header">United States</td>
</tr>
<tr>
<td class="ic3-mdx-result-cell-header">New Value</td>
 <td>2</td>
</tr>
</table>



 
</div>

           

<div class="ic3-mdx-example-header ic3-mdx-example-header-collapsed" onclick="ic3_mdx_example_click(this)">

#### Market Share

</div>

<div class="ic3-mdx-example-body" style="display:none">

A classical example showing how to calculate market shares.

Market Share X = Licenses sold of type X / Total number of licences sold

In the example, you can see how to combine in the same expression tuples with different dimensionality.

**Query**

```
WITH
    MEMBER [Measures].[Market Share] AS ([Measures].[Amount]) / ( [Product].currentMember.parent ,[Measures].[Amount] ), FORMAT_STRING = "Percent"
SELECT
    [Measures].[Market Share] ON 0,
    [Product].[icCube].children ON 1
FROM
    [Sales]
WHERE
    [Geography].[Geo].[Europe]
```

 
**Result**

<table  class="ic3-mdx-result">
<tr>
  <td class="ic3-mdx-result-cell-empty"></td>
  <td class="ic3-mdx-result-cell-header">Market Share</td>
</tr>
<tr>
<td class="ic3-mdx-result-cell-header">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Corporate</td>
 <td>56.47%</td>
</tr>
<tr>
<td class="ic3-mdx-result-cell-header">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Partnership</td>
 <td>37.65%</td>
</tr>
<tr>
<td class="ic3-mdx-result-cell-header">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Personal</td>
 <td>5.88%</td>
</tr>
<tr>
<td class="ic3-mdx-result-cell-header">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Startup</td>
 <td></td>
</tr>
</table>



 
</div>

           

<div class="ic3-mdx-example-header ic3-mdx-example-header-collapsed" onclick="ic3_mdx_example_click(this)">

#### Market Share with Query Cache

</div>

<div class="ic3-mdx-example-body" style="display:none">

The former example with the query cache activated. This is useful if the query is using the same tuple multiple time in
the same query (this is not the case for the example)

**Query**

```
WITH
    MEMBER [Measures].[Market Share] AS ([Measures].[Amount]) / ( [Product].currentMember.parent ,[Measures].[Amount] ), REQUEST_CACHED = true
SELECT
    [Measures].[Market Share] ON 0,
    [Product].[icCube].children ON 1
FROM
    [Sales]
WHERE
    [Geography].[Geo].[Europe]
```

 
**Result**

<table  class="ic3-mdx-result">
<tr>
  <td class="ic3-mdx-result-cell-empty"></td>
  <td class="ic3-mdx-result-cell-header">Market Share</td>
</tr>
<tr>
<td class="ic3-mdx-result-cell-header">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Corporate</td>
 <td>0.5647058823529412</td>
</tr>
<tr>
<td class="ic3-mdx-result-cell-header">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Partnership</td>
 <td>0.3764705882352941</td>
</tr>
<tr>
<td class="ic3-mdx-result-cell-header">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Personal</td>
 <td>0.058823529411764705</td>
</tr>
<tr>
<td class="ic3-mdx-result-cell-header">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Startup</td>
 <td></td>
</tr>
</table>



 
</div>

           

<div class="ic3-mdx-example-header ic3-mdx-example-header-collapsed" onclick="ic3_mdx_example_click(this)">

#### Overwriting members

</div>

<div class="ic3-mdx-example-body" style="display:none">

In this example, we show how to overwrite the current member in a calculated member.

Even though we are requesting the value for [United States] on the axis 0, the calculated member is using the value for [Switzerland] as the hierarchy member is redefined for the current tuple (in this example coming from a cell evaluation).

**Query**

```
WITH
    MEMBER [Measures].[Switzerland Value] AS ( [Geography].[Geo].[Switzerland],[Measures].[Amount] )
SELECT
    { [Geography].[Geo].[Switzerland],[Geography].[Geo].[United States] } ON 0,
    [Measures].[Switzerland Value] ON 1
FROM
    [Sales]
```

 
**Result**

<table  class="ic3-mdx-result">
<tr>
  <td class="ic3-mdx-result-cell-empty"></td>
  <td class="ic3-mdx-result-cell-header">Switzerland</td>
  <td class="ic3-mdx-result-cell-header">United States</td>
</tr>
<tr>
<td class="ic3-mdx-result-cell-header">Switzerland Value</td>
 <td>248</td>
 <td>248</td>
</tr>
</table>



**Assertion : Cell Equals**

```
SELECT
    { [Geography].[Geo].[Switzerland],[Geography].[Geo].[Switzerland] } ON 0,
    [Measures].[Amount] ON 1
FROM
    [Sales]
```


 
</div>

           

<div class="ic3-mdx-example-header ic3-mdx-example-header-collapsed" onclick="ic3_mdx_example_click(this)">

#### Using current member

</div>

<div class="ic3-mdx-example-body" style="display:none">

We can ask for the current member of a hierarchy and act in consequence.

In this example we define a measure which is changing the value only for [Switzerland].

**Query**

```
WITH
    MEMBER [Measures].[Switzerland as 2] AS IIF([Geography].[Geo].currentMember is [Geography].[Geo].[Switzerland], 2, [Measures].[Amount] )
SELECT
    { [Geography].[Geo].[Switzerland], [Geography].[Geo].[United States] } ON 0,
    [Measures].[Switzerland as 2] ON 1
FROM
    [Sales]
```

 
**Result**

<table  class="ic3-mdx-result">
<tr>
  <td class="ic3-mdx-result-cell-empty"></td>
  <td class="ic3-mdx-result-cell-header">Switzerland</td>
  <td class="ic3-mdx-result-cell-header">United States</td>
</tr>
<tr>
<td class="ic3-mdx-result-cell-header">Switzerland as 2</td>
 <td>2</td>
 <td>768</td>
</tr>
</table>



**Assertion : Cell Equals**

```
WITH
    MEMBER [Measures].[2] AS 2
SELECT
    { [Measures].[2], [Measures].[Amount] } ON 0,
    [Geography].[Geo].[United States] ON 1
FROM
    [Sales]
```


 
</div>

           

<div class="ic3-mdx-example-header ic3-mdx-example-header-collapsed" onclick="ic3_mdx_example_click(this)">

#### Other than [Measures] dimension

</div>

<div class="ic3-mdx-example-body" style="display:none">

Calculated measures are not limited to the [Measures] dimension. New members (hence the name calculated members) can be added into any existing hierarchy.

It's important to note that as for [Measures] the calculated members are not added into the original hierarchy structure. This implies that functions like Children() and Members() will not return calculated members.

**Query**

```
WITH
    // Calculates the average over [Europe] countries, new member is added without level and parent
    MEMBER [Geography].[Geo].[Europe Avg] AS Avg( [Geography].[Geo].[Europe].children )
    // Calculates the max for all [Europe] countries, new member added as a child of a [Europe]
    MEMBER [Geography].[Geo].[Europe].[Europe Max] AS Max( [Geography].[Geo].[Europe].children)
    // Difference between [Europe] and [Switzerland], new member added as a child of a country
    MEMBER [Geography].[Geo].[Europe].[Europe - Switzerland] AS [Geography].[Geo].[Europe] - [Geography].[Geo].[Switzerland]
SELECT
    { [Geography].[Geo].[Europe], [Geography].[Geo].[Europe Avg], [Geography].[Geo].[Europe Max], [Geography].[Geo].[Europe - Switzerland],[Geography].[Geo].[Europe].children} ON 0,
    { [Measures].[Amount] } ON 1
FROM
    [Sales]
```

 
**Result**

<table  class="ic3-mdx-result">
<tr>
  <td class="ic3-mdx-result-cell-empty"></td>
  <td class="ic3-mdx-result-cell-header">Europe</td>
  <td class="ic3-mdx-result-cell-header">Europe Avg</td>
  <td class="ic3-mdx-result-cell-header">Europe Max</td>
  <td class="ic3-mdx-result-cell-header">Europe - Switzerland</td>
  <td class="ic3-mdx-result-cell-header">France</td>
  <td class="ic3-mdx-result-cell-header">Spain</td>
  <td class="ic3-mdx-result-cell-header">Switzerland</td>
</tr>
<tr>
<td class="ic3-mdx-result-cell-header">Amount</td>
 <td>255</td>
 <td>85.0</td>
 <td>248</td>
 <td>7</td>
 <td>4</td>
 <td>3</td>
 <td>248</td>
</tr>
</table>



**Assertion : Cell Equals**

```
WITH
    MEMBER [Geography].[Geo].[Europe Avg] AS Avg( [Geography].[Geo].[Europe].children )
    MEMBER [Geography].[Geo].[Europe Max] AS Max( [Geography].[Geo].[Europe].children)
    MEMBER [Geography].[Geo].[Europe - Switzerland] AS [Geography].[Geo].[Europe] - [Geography].[Geo].[Switzerland]
SELECT
    { [Geography].[Geo].[Europe], [Geography].[Geo].[Europe Avg], [Geography].[Geo].[Europe Max], [Geography].[Geo].[Europe - Switzerland],[Geography].[Geo].[Europe].children} ON 0,
    { [Measures].[Amount] } ON 1
FROM
    [Sales]
```


 
</div>

           

<div class="ic3-mdx-example-header ic3-mdx-example-header-collapsed" onclick="ic3_mdx_example_click(this)">

#### Current Member

</div>

<div class="ic3-mdx-example-body" style="display:none">

The calculated member/measure being evaluated is the .CurrentMember().

**Query**

```
WITH
    MEMBER [Measures].[Y] AS [Measures].CurrentMember.NAME
    MEMBER [Measures].[X] AS [Measures].CurrentMember.NAME + '-' + [Measures].[Y]
SELECT
    [Measures].[X] ON 0
FROM
    [Sales]
```

 
**Result**

<table  class="ic3-mdx-result">
<tr>
  <td class="ic3-mdx-result-cell-header">X</td>
</tr>
<tr>
 <td>X-Y</td>
</tr>
</table>



**Assertion : MDX Equals**

```
WITH
    MEMBER [Measures].[X] AS "X-Y"
SELECT
    [Measures].[X] ON 0
FROM
    [Sales]
```


 
</div>

           

<div class="ic3-mdx-example-header ic3-mdx-example-header-collapsed" onclick="ic3_mdx_example_click(this)">

#### Overwriting members

</div>

<div class="ic3-mdx-example-body" style="display:none">

Overwriting an existing member is not possible. However it's possible to add a member with the same name in a different path.

In this example we are adding [Switzerland] in the root of the [Geo] hierarchy.

**Query**

```
WITH
    MEMBER [Geography].[Geo].[Swiss-Lausanne]   AS ([Geography].[Geo].[Switzerland] /* calculated member */)
    MEMBER [Geography].[Geo].[Swiss]            AS ([Geography].[Geo].[Europe].[Switzerland] /* hierarchy member */ )
    MEMBER [Geography].[Geo].[Switzerland]      AS ([Geography].[Geo].[Lausanne])
SELECT
    { [Geography].[Geo].[Switzerland],[Geography].[Geo].[Swiss], [Geography].[Geo].[Swiss-Lausanne] } ON 0
FROM
    [Sales]
```

 
**Result**

<table  class="ic3-mdx-result">
<tr>
  <td class="ic3-mdx-result-cell-header">Switzerland</td>
  <td class="ic3-mdx-result-cell-header">Swiss</td>
  <td class="ic3-mdx-result-cell-header">Swiss-Lausanne</td>
</tr>
<tr>
 <td>56</td>
 <td>248</td>
 <td>56</td>
</tr>
</table>



**Assertion : Cell Equals**

```
SELECT
    { [Geography].[Geo].[Lausanne],[Geography].[Geo].[Europe].[Switzerland], [Geography].[Geo].[Lausanne] } ON 0
FROM
    [Sales]
```


 
</div>

           

<div class="ic3-mdx-example-header ic3-mdx-example-header-collapsed" onclick="ic3_mdx_example_click(this)">

#### Overwriting existing members - Error

</div>

<div class="ic3-mdx-example-body" style="display:none">

Overwriting an existing member is not possible (if same path).

**Query**

```
WITH
    MEMBER [Geography].[Geo].[Europe].[Switzerland] AS ([Geography].[Geo].[Lausanne])
SELECT
    { [Geography].[Geo].[Switzerland] } ON 0
FROM
    [Sales]
```

 
**Result**

<table class="ic3-mdx-result"><tr><td>Error Code</td><td>

[OLAP_CALC_MEMBER_OVERWRITE_MEMBER](icCubeError.md#olap_calc_member_overwrite_member)</td></tr><tr><td>Error Message</td><td>

the calc. member 'Switzerland' is overwriting an existing member '[Geography].[Geo].[Country].&[CH]'</td></tr></table>


 
</div>

           

<div class="ic3-mdx-example-header ic3-mdx-example-header-collapsed" onclick="ic3_mdx_example_click(this)">

#### Tuple completion - same hierarchy

</div>

<div class="ic3-mdx-example-body" style="display:none">

Within the calculated measure the tuple is completed with hierarchies coming from the axis.

In our example, the [Economy] member coming from axis 1 is 'set' in the calculated member.

**Query**

```
WITH
    MEMBER [Measures].[xx] AS ( [Geography].[Geo].[France], [Measures].[Amount] )
SELECT
    [Measures].[xx] on 0,
    { [Geography].[Economy].[France], [Geography].[Economy].[Spain] } on 1
FROM
    [Sales]
```

 
**Result**

<table  class="ic3-mdx-result">
<tr>
  <td class="ic3-mdx-result-cell-empty"></td>
  <td class="ic3-mdx-result-cell-header">xx</td>
</tr>
<tr>
<td class="ic3-mdx-result-cell-header">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;France</td>
 <td>4</td>
</tr>
<tr>
<td class="ic3-mdx-result-cell-header">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Spain</td>
 <td></td>
</tr>
</table>



**Assertion : Cell Equals**

```
WITH
    MEMBER [Measures].[xx] AS ( [Geography].[Geo].[France], [Geography].[Economy].currentMember, [Measures].[Amount] )
SELECT
    [Measures].[xx] on 0,
    { [Geography].[Economy].[France], [Geography].[Economy].[Spain] } on 1
FROM
    [Sales]
```


 
</div>

           

<div class="ic3-mdx-example-header ic3-mdx-example-header-collapsed" onclick="ic3_mdx_example_click(this)">

#### Tuple completion - iteration

</div>

<div class="ic3-mdx-example-body" style="display:none">

Within the calculated measure the [Geo] hierarchy is completed by the current context.

In our example, the calculated member tuple ([Geography].[Geo].[France],[Measures].[Amount]) is completed with the member [Geography].[Economy].[Spain] coming from the axis.

**Query**

```
WITH
    MEMBER [Measures].[xx] AS Iif( ( [Geography].[Geo].[France], [Measures].[Amount] ) = 4 , 'exist', 'no' )
SELECT
    Iif( Generate( { [Geography].[Economy].[Spain] }, [Measures].[xx] ) = 'exist', [Time].[Calendar].[2010], [Time].[Calendar].[2011] ) on 0
FROM
    [Sales]
```

 
**Result**

<table  class="ic3-mdx-result">
<tr>
  <td class="ic3-mdx-result-cell-header">2011</td>
</tr>
<tr>
 <td></td>
</tr>
</table>



**Assertion : MDX Equals**

```
SELECT [Time].[Calendar].[2011] ON 0 FROM [Sales]
```


 
</div>

           

<div class="ic3-mdx-example-header ic3-mdx-example-header-collapsed" onclick="ic3_mdx_example_click(this)">

#### Recursion : current member

</div>

<div class="ic3-mdx-example-body" style="display:none">

The calculated measures being evaluated is the .CurrentMember() so beware of infinite loop.

Here [X] as current member of the [Product].[Prod] hierarchy in its formula is creating an infinite loop.

Why is it a recursion ?

( [Product].[Prod].[X] )
is evaluated as ( [Measures].[Amount] )
is evaluated as ( [Measures].[Amount], [Product].[Prod].currentMember )
is evaluated as ( [Measures].[Amount], [Product].[Prod].[X] )
is evaluated as ( [Product].[Prod].[X] )

the latest one defines a recursion.

**Query**

```
WITH
    MEMBER [Product].[Prod].[X] AS [Measures].[Amount]
SELECT
    [Product].[Prod].[X] ON 0
FROM
    [Sales]
```

 
**Result**

<table  class="ic3-mdx-result">
<tr>
  <td class="ic3-mdx-result-cell-header">X</td>
</tr>
<tr>
 <td>#ERROR : (OLAP_CALC_MEMBER_STACK_OVERFLOW)</td>
</tr>
</table>
<pre>Error Code    : [OLAP_CALC_MEMBER_STACK_OVERFLOW](icCubeError.md#olap_calc_member_stack_overflow)<br>Error Message : Calculated member [Product].[Prod].[Company].[X] stack overflow (see icCube.calcMemberMaxEvaluationFrameStackSize XMLA property) detected at (X X)</pre>


 
</div>

           

<div class="ic3-mdx-example-header ic3-mdx-example-header-collapsed" onclick="ic3_mdx_example_click(this)">

#### Recursion : current member (2)

</div>

<div class="ic3-mdx-example-body" style="display:none">

The calculated measures being evaluated is the .CurrentMember() so beware of infinite loop.

Here [X] as current member of the [Product].[Prod] hierarchy in its formula is creating an infinite loop.

**Query**

```
WITH
    MEMBER [Product].[Prod].[X] AS Avg( [Geography].[Geo].Members )
SELECT
    [Product].[Prod].[X] ON 0
FROM
    [Sales]
```

 
**Result**

<table  class="ic3-mdx-result">
<tr>
  <td class="ic3-mdx-result-cell-header">X</td>
</tr>
<tr>
 <td>#ERROR : (OLAP_CALC_MEMBER_STACK_OVERFLOW)</td>
</tr>
</table>
<pre>Error Code    : [OLAP_CALC_MEMBER_STACK_OVERFLOW](icCubeError.md#olap_calc_member_stack_overflow)<br>Error Message : Calculated member [Product].[Prod].[Company].[X] stack overflow (see icCube.calcMemberMaxEvaluationFrameStackSize XMLA property) detected at (X X)</pre>


 
</div>

           

<div class="ic3-mdx-example-header ic3-mdx-example-header-collapsed" onclick="ic3_mdx_example_click(this)">

#### Children

</div>

<div class="ic3-mdx-example-body" style="display:none">

A calculated member can contains other calculated members. The function children is returning the calculated members.

**Query**

```
WITH
    MEMBER [Geography].[Geo].[All Regions].[XX]      as 42
    MEMBER [Geography].[Geo].[All Regions].[XX].[X1] as 10
    MEMBER [Geography].[Geo].[All Regions].[XX].[X2] as 20

SELECT
    [Geography].[Geo].[All Regions].[XX].children ON 0
FROM
    [Sales]
```

 
**Result**

<table  class="ic3-mdx-result">
<tr>
  <td class="ic3-mdx-result-cell-header">X1</td>
  <td class="ic3-mdx-result-cell-header">X2</td>
</tr>
<tr>
 <td>10</td>
 <td>20</td>
</tr>
</table>



**Assertion : Cell Equals**

```
WITH
    MEMBER [Geography].[Geo].[All Regions].[XX]      as 42
    MEMBER [Geography].[Geo].[All Regions].[XX].[X1] as 10
    MEMBER [Geography].[Geo].[All Regions].[XX].[X2] as 20

SELECT
    { [Geography].[Geo].[All Regions].[XX].[X1], [Geography].[Geo].[All Regions].[XX].[X2] } ON 0
FROM
    [Sales]
```


 
</div>

           

<div class="ic3-mdx-example-header ic3-mdx-example-header-collapsed" onclick="ic3_mdx_example_click(this)">

#### Implicit anchor

</div>

<div class="ic3-mdx-example-body" style="display:none">

A calculated member defined outside any hierarchy is implicitly defined within the [Measures] dimension.

**Query**

```
WITH
    MEMBER [the-answer] AS 42
    MEMBER [the-answer-hierarchy] AS [the-answer].Hierarchy.Name

SELECT
    { [the-answer], [the-answer-hierarchy] } ON 0
FROM
    [Sales]
```

 
**Result**

<table  class="ic3-mdx-result">
<tr>
  <td class="ic3-mdx-result-cell-header">the-answer</td>
  <td class="ic3-mdx-result-cell-header">the-answer-hierarchy</td>
</tr>
<tr>
 <td>42</td>
 <td>Measures</td>
</tr>
</table>



**Assertion : MDX Equals**

```
WITH
    MEMBER [Measures].[the-answer] AS 42
    MEMBER [Measures].[the-answer-hierarchy] AS "Measures"

SELECT
    { [Measures].[the-answer], [Measures].[the-answer-hierarchy] } ON 0
FROM
    [Sales]
```


 
</div>

           

<div class="ic3-mdx-example-header ic3-mdx-example-header-collapsed" onclick="ic3_mdx_example_click(this)">

#### Level anchor

</div>

<div class="ic3-mdx-example-body" style="display:none">

A calculated member can be defined directly into a level of depth zero.

**Query**

```
WITH MEMBER [Product].[Prod].[Company].[New Company] AS 42
SELECT FROM [Sales] WHERE [Product].[Prod].[Company].[New Company]
```

 
**Result**

<pre>42</pre>



**Assertion : Cell Equals**

```
WITH MEMBER [Expected Result] AS 42
SELECT FROM [Sales] WHERE [Expected Result]
```


 
</div>

           

<div class="ic3-mdx-example-header ic3-mdx-example-header-collapsed" onclick="ic3_mdx_example_click(this)">

#### Level anchor - orphan

</div>

<div class="ic3-mdx-example-body" style="display:none">

A calculated member cannot be defined into a level of depth greater than zero without
specifying any parent in the previous level.

**Query**

```
WITH MEMBER [Product].[Prod].[Article].[OEM] AS 42
SELECT FROM [Sales] WHERE [Product].[Prod].[Article].[OEM]
```

 
**Result**

<table class="ic3-mdx-result"><tr><td>Error Code</td><td>

[OLAP_CALC_MEMBER_ORPHAN](icCubeError.md#olap_calc_member_orphan)</td></tr><tr><td>Error Message</td><td>

cannot attach the calculated member 'OEM' into the level '[Product].[Prod].[Article]' without defining any parent (hint: attach the calculated member to a member of the previous level)</td></tr></table>


 
</div>

           

<div class="ic3-mdx-example-header ic3-mdx-example-header-collapsed" onclick="ic3_mdx_example_click(this)">

#### Recursion : infinite loop

</div>

<div class="ic3-mdx-example-body" style="display:none">

Recursion is supported but beware of infinite loop.

**Query**

```
WITH
    MEMBER [Product].[Prod].[X] AS [Product].[Prod].[X]
SELECT
    [Product].[Prod].[X] ON 0
FROM
    [Sales]
```

 
**Result**

<table  class="ic3-mdx-result">
<tr>
  <td class="ic3-mdx-result-cell-header">X</td>
</tr>
<tr>
 <td>#ERROR : (OLAP_CALC_MEMBER_STACK_OVERFLOW)</td>
</tr>
</table>
<pre>Error Code    : [OLAP_CALC_MEMBER_STACK_OVERFLOW](icCubeError.md#olap_calc_member_stack_overflow)<br>Error Message : Calculated member [Product].[Prod].[Company].[X] stack overflow (see icCube.calcMemberMaxEvaluationFrameStackSize XMLA property) detected at (X X)</pre>


 
</div>

           

<div class="ic3-mdx-example-header ic3-mdx-example-header-collapsed" onclick="ic3_mdx_example_click(this)">

#### Recursion : infinite loop (2)

</div>

<div class="ic3-mdx-example-body" style="display:none">

Recursion is supported but beware of infinite loop.

**Query**

```
WITH
    MEMBER [Product].[Prod].[X] AS 1 + [Product].[Prod].[X]
SELECT
    [Product].[Prod].[X] ON 0
FROM
    [Sales]
```

 
**Result**

<table  class="ic3-mdx-result">
<tr>
  <td class="ic3-mdx-result-cell-header">X</td>
</tr>
<tr>
 <td>#ERROR : (OLAP_CALC_MEMBER_STACK_OVERFLOW)</td>
</tr>
</table>
<pre>Error Code    : [OLAP_CALC_MEMBER_STACK_OVERFLOW](icCubeError.md#olap_calc_member_stack_overflow)<br>Error Message : Calculated member [Product].[Prod].[Company].[X] stack overflow (see icCube.calcMemberMaxEvaluationFrameStackSize XMLA property) detected at (X X)</pre>


 
</div>

           

<div class="ic3-mdx-example-header ic3-mdx-example-header-collapsed" onclick="ic3_mdx_example_click(this)">

#### Recursion : infinite loop (3)

</div>

<div class="ic3-mdx-example-body" style="display:none">

Recursion is supported but beware of infinite loop.

**Query**

```
WITH
    MEMBER [Measures].[Z] AS [Measures].[X]
    MEMBER [Measures].[Y] AS [Measures].[Z]
    MEMBER [Measures].[X] AS [Measures].[Y]
SELECT
    [Measures].[X] ON 0
FROM
    [Sales]
```

 
**Result**

<table  class="ic3-mdx-result">
<tr>
  <td class="ic3-mdx-result-cell-header">X</td>
</tr>
<tr>
 <td>#ERROR : (OLAP_CALC_MEMBER_STACK_OVERFLOW)</td>
</tr>
</table>
<pre>Error Code    : [OLAP_CALC_MEMBER_STACK_OVERFLOW](icCubeError.md#olap_calc_member_stack_overflow)<br>Error Message : Calculated member [Measures].[Z] stack overflow (see icCube.calcMemberMaxEvaluationFrameStackSize XMLA property) detected at (Z Y X Z)</pre>


 
</div>

                                                                                                            

### See Also


[Cell Properties](Cell%20Properties.md)

[Create (Member)](Create%20(Member).md)

[Drop (Member)](Drop%20(Member).md)

[AddCalculatedMembers](AddCalculatedMembers.md)

[StripCalculatedMembers](StripCalculatedMembers.md)


### XMLA Properties

Find [here](icCubeDefaultXmlaProperties.md) the value of the XMLA properties when not specified otherwise.

_
